/*
 * Decompiled with CFR 0.152.
 */
package org.xtreemfs.babudb.index.reader;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.xtreemfs.babudb.api.database.ResultSet;
import org.xtreemfs.babudb.api.index.ByteRangeComparator;
import org.xtreemfs.babudb.index.ByteRange;
import org.xtreemfs.babudb.index.reader.BlockReader;
import org.xtreemfs.babudb.index.reader.CompressedBlockReader;
import org.xtreemfs.babudb.index.reader.DefaultBlockReader;
import org.xtreemfs.babudb.index.reader.DiskIndexIterator;
import org.xtreemfs.babudb.index.reader.InternalDiskIndexIterator;
import org.xtreemfs.babudb.index.reader.SearchUtil;
import org.xtreemfs.foundation.logging.Logging;

public class DiskIndex {
    private ByteBuffer blockIndexBuf;
    private BlockReader blockIndex;
    private MappedByteBuffer[] dbFiles;
    private FileChannel[] dbFileChannels;
    private ByteRangeComparator comp;
    private long indexSize;
    private final boolean compressed;
    private final boolean mmaped;

    public DiskIndex(String path, ByteRangeComparator comp, boolean compressed, boolean mmaped) throws IOException {
        if (!path.endsWith(System.getProperty("file.separator"))) {
            path = path + System.getProperty("file.separator");
        }
        if (!new File(path).exists()) {
            throw new IOException("There is no index at " + path);
        }
        this.comp = comp;
        this.compressed = compressed;
        this.mmaped = mmaped;
        Logging.logMessage((int)6, (Object)this, (String)"loading index ...", (Object[])new Object[0]);
        RandomAccessFile blockIndexFile = new RandomAccessFile(path + "blockindex.idx", "r");
        this.blockIndexBuf = ByteBuffer.allocate((int)blockIndexFile.length());
        FileChannel channel = blockIndexFile.getChannel();
        channel.read(this.blockIndexBuf);
        this.blockIndex = new DefaultBlockReader(this.blockIndexBuf, 0, this.blockIndexBuf.limit(), comp);
        channel.close();
        FilenameFilter filter = new FilenameFilter(){

            @Override
            public boolean accept(File dir, String filename) {
                return filename.startsWith("blockfile_");
            }
        };
        String[] blockFilenames = new File(path).list(filter);
        Pattern p = Pattern.compile("blockfile_(\\d+).idx");
        this.dbFileChannels = new FileChannel[blockFilenames.length];
        if (mmaped) {
            this.dbFiles = new MappedByteBuffer[blockFilenames.length];
        }
        for (String blockFilename : blockFilenames) {
            Matcher m = p.matcher(blockFilename);
            if (!m.matches()) continue;
            int blockIndexId = new Integer(m.group(1));
            RandomAccessFile blockFile = new RandomAccessFile(path + blockFilename, "r");
            this.dbFileChannels[blockIndexId] = blockFile.getChannel();
            this.indexSize += blockFile.length();
            if (!mmaped) continue;
            this.dbFiles[blockIndexId] = this.dbFileChannels[blockIndexId].map(FileChannel.MapMode.READ_ONLY, 0L, blockFile.length());
            Logging.logMessage((int)6, (Object)this, (String)("block file index size: " + blockFile.length()), (Object[])new Object[0]);
            this.dbFileChannels[blockIndexId].close();
        }
    }

    public byte[] lookup(byte[] key) {
        int endBlockOffset;
        int indexPosition = this.getBlockIndexPosition(key, this.blockIndex);
        if (indexPosition == -1) {
            return null;
        }
        int startBlockOffset = DiskIndex.getBlockOffset(indexPosition, this.blockIndex);
        short fileId = DiskIndex.getBlockFileId(indexPosition, this.blockIndex);
        if (indexPosition == this.blockIndex.getNumEntries() - 1) {
            endBlockOffset = -1;
        } else {
            ByteRange indexPos = DiskIndex.getBlockEntry(indexPosition + 1, this.blockIndex);
            ByteBuffer indexPosBuf = indexPos.getBuf();
            endBlockOffset = DiskIndex.getBlockIndexOffset(indexPosBuf, indexPos.getStartOffset());
            if (DiskIndex.getBlockIndexFileId(indexPosBuf, indexPos.getStartOffset()) > fileId) {
                endBlockOffset = -1;
            }
        }
        BlockReader targetBlock = null;
        try {
            targetBlock = this.mmaped ? this.getBlock(startBlockOffset, endBlockOffset, this.dbFiles[fileId]) : this.getBlock(startBlockOffset, endBlockOffset, this.dbFileChannels[fileId]);
        }
        catch (IOException e) {
            Logging.logError((int)3, (Object)this, (Throwable)e);
        }
        ByteRange val = targetBlock.lookup(key);
        byte[] result = val == null ? null : val.toBuffer();
        targetBlock.free();
        return result;
    }

    public long numKeys() {
        int numBlocks = this.blockIndex.getNumEntries();
        if (numBlocks == 0) {
            return 0L;
        }
        int lastBlockStartOffset = DiskIndex.getBlockOffset(numBlocks - 1, this.blockIndex);
        int lastBlockEndOffset = -1;
        BlockReader lastBlock = null;
        try {
            lastBlock = this.mmaped ? this.getBlock(lastBlockStartOffset, lastBlockEndOffset, this.dbFiles[DiskIndex.getBlockFileId(numBlocks - 1, this.blockIndex)]) : this.getBlock(lastBlockStartOffset, lastBlockEndOffset, this.dbFileChannels[DiskIndex.getBlockFileId(numBlocks - 1, this.blockIndex)]);
        }
        catch (IOException e) {
            Logging.logError((int)3, (Object)this, (Throwable)e);
        }
        long lastBlockEntryCount = lastBlock.getNumEntries();
        lastBlock.free();
        if (numBlocks == 1) {
            return lastBlockEntryCount;
        }
        int firstBlockStartOffset = 0;
        int firstBlockEndBlockOffset = DiskIndex.getBlockOffset(1, this.blockIndex);
        if (DiskIndex.getBlockFileId(1, this.blockIndex) > DiskIndex.getBlockFileId(0, this.blockIndex)) {
            firstBlockEndBlockOffset = -1;
        }
        BlockReader firstBlock = null;
        try {
            firstBlock = this.mmaped ? this.getBlock(firstBlockStartOffset, firstBlockEndBlockOffset, this.dbFiles[DiskIndex.getBlockFileId(0, this.blockIndex)]) : this.getBlock(firstBlockStartOffset, firstBlockEndBlockOffset, this.dbFileChannels[DiskIndex.getBlockFileId(0, this.blockIndex)]);
        }
        catch (IOException e) {
            Logging.logError((int)3, (Object)this, (Throwable)e);
        }
        long firstBlocksEntryCount = (long)firstBlock.getNumEntries() * (long)(numBlocks - 1);
        firstBlock.free();
        return firstBlocksEntryCount + lastBlockEntryCount;
    }

    public ResultSet<byte[], byte[]> rangeLookup(byte[] from, byte[] to, boolean ascending) {
        if (this.mmaped) {
            ByteBuffer[] map = new ByteBuffer[this.dbFiles.length];
            for (int i = 0; i < this.dbFiles.length; ++i) {
                this.dbFiles[i].position(0);
                map[i] = this.dbFiles[i].slice();
            }
            return new DiskIndexIterator(this, this.blockIndex, from, to, ascending, map);
        }
        return new DiskIndexIterator(this, this.blockIndex, from, to, ascending, this.dbFileChannels);
    }

    public InternalDiskIndexIterator internalRangeLookup(byte[] from, byte[] to, boolean ascending) {
        if (this.mmaped) {
            ByteBuffer[] map = new ByteBuffer[this.dbFiles.length];
            for (int i = 0; i < this.dbFiles.length; ++i) {
                this.dbFiles[i].position(0);
                map[i] = this.dbFiles[i].slice();
            }
            return new InternalDiskIndexIterator(this, this.blockIndex, from, to, ascending, map);
        }
        return new InternalDiskIndexIterator(this, this.blockIndex, from, to, ascending, this.dbFileChannels);
    }

    public ByteRangeComparator getComparator() {
        return this.comp;
    }

    public long getSize() {
        return this.indexSize;
    }

    public void destroy() throws IOException {
        this.blockIndex.free();
        for (FileChannel c : this.dbFileChannels) {
            c.close();
        }
    }

    public void finalize() throws Throwable {
        try {
            this.destroy();
        }
        catch (IOException exc) {
            Logging.logError((int)3, (Object)this, (Throwable)exc);
        }
        super.finalize();
    }

    protected BlockReader getBlock(int startBlockOffset, int endBlockOffset, ByteBuffer map) {
        if (startBlockOffset > map.limit()) {
            return null;
        }
        if (endBlockOffset == -1) {
            endBlockOffset = map.limit();
        }
        BlockReader targetBlock = this.compressed ? new CompressedBlockReader(map, startBlockOffset, endBlockOffset, this.comp) : new DefaultBlockReader(map, startBlockOffset, endBlockOffset, this.comp);
        return targetBlock;
    }

    protected BlockReader getBlock(int startBlockOffset, int endBlockOffset, FileChannel channel) throws IOException {
        if ((long)startBlockOffset > channel.size()) {
            return null;
        }
        if (endBlockOffset == -1) {
            endBlockOffset = (int)channel.size();
        }
        BlockReader targetBlock = this.compressed ? new CompressedBlockReader(channel, startBlockOffset, endBlockOffset, this.comp) : new DefaultBlockReader(channel, startBlockOffset, endBlockOffset, this.comp);
        return targetBlock;
    }

    protected int getBlockIndexPosition(byte[] key, BlockReader index) {
        return SearchUtil.getInclBottomOffset(index.getKeys(), key, this.comp);
    }

    protected static int getBlockOffset(int indexPosition, BlockReader index) {
        ByteRange range = index.getValues().getEntry(indexPosition);
        return range.getBuf().getInt(range.getStartOffset());
    }

    protected static short getBlockFileId(int indexPosition, BlockReader index) {
        ByteRange range = index.getValues().getEntry(indexPosition);
        return range.getBuf().getShort(range.getStartOffset() + 4);
    }

    protected static ByteRange getBlockEntry(int indexPosition, BlockReader index) {
        ByteRange range = index.getValues().getEntry(indexPosition);
        return range;
    }

    protected static int getBlockIndexOffset(ByteBuffer buf, int startOffset) {
        return buf.getInt(startOffset);
    }

    protected static short getBlockIndexFileId(ByteBuffer buf, int startOffset) {
        return buf.getShort(startOffset + 4);
    }
}

