/*
 * Decompiled with CFR 0.152.
 */
package org.xtreemfs.babudb.lsmdb;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.channels.ClosedByInterruptException;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.xtreemfs.babudb.api.exception.BabuDBException;
import org.xtreemfs.babudb.api.index.ByteRangeComparator;
import org.xtreemfs.babudb.index.LSMTree;
import org.xtreemfs.babudb.lsmdb.LSN;
import org.xtreemfs.babudb.snapshots.SnapshotConfig;
import org.xtreemfs.foundation.logging.Logging;
import org.xtreemfs.foundation.util.FSUtils;

public class LSMDatabase {
    public static final int MAX_INDICES = 512;
    public static final LSN NO_DB_LSN = new LSN(0, 0L);
    private static final String SNAPSHOT_FILENAME_REGEXP = "IX(\\d+)V(\\d+)SEQ(\\d+)\\.idx";
    private final List<LSMTree> trees;
    private final String databaseDir;
    private final String databaseName;
    private final int databaseId;
    private LSN ondiskLSN;
    private final int numIndices;
    private final ByteRangeComparator[] comparators;
    private final boolean compression;
    private final int maxEntriesPerBlock;
    private final int maxBlockFileSize;
    private final boolean disableMMap;
    private final int mmapLimit;

    public LSMDatabase(String databaseName, int databaseId, String databaseDir, int numIndices, boolean readFromDisk, ByteRangeComparator[] comparators, boolean compression, int maxEntriesPerBlock, int maxBlockFileSize, boolean disableMMap, int mmapLimit) throws BabuDBException {
        this.numIndices = numIndices;
        this.databaseId = databaseId;
        File f = new File(databaseDir);
        if (!f.exists()) {
            f.mkdirs();
        }
        this.databaseDir = f.getAbsolutePath();
        this.databaseName = databaseName;
        this.trees = new ArrayList<LSMTree>(numIndices);
        this.comparators = comparators;
        this.compression = compression;
        this.maxEntriesPerBlock = maxEntriesPerBlock;
        this.maxBlockFileSize = maxBlockFileSize;
        this.disableMMap = disableMMap;
        this.mmapLimit = mmapLimit;
        if (readFromDisk) {
            this.loadFromDisk(numIndices);
        } else {
            try {
                for (int i = 0; i < numIndices; ++i) {
                    assert (comparators[i] != null);
                    this.trees.add(new LSMTree(null, comparators[i], this.compression, maxEntriesPerBlock, maxBlockFileSize, !disableMMap, mmapLimit));
                }
                this.ondiskLSN = NO_DB_LSN;
            }
            catch (IOException ex) {
                throw new BabuDBException(BabuDBException.ErrorCode.IO_ERROR, "cannot create new index", ex);
            }
        }
    }

    public String[] getComparatorClassNames() {
        String[] array = new String[this.trees.size()];
        for (int i = 0; i < this.trees.size(); ++i) {
            array[i] = this.comparators[i].getClass().getName();
        }
        return array;
    }

    public ByteRangeComparator[] getComparators() {
        return this.comparators;
    }

    private void loadFromDisk(int numIndices) throws BabuDBException {
        int index;
        Logging.logMessage((int)7, (Object)this, (String)("loading database " + this.databaseName + " from disk..."), (Object[])new Object[0]);
        for (index = 0; index < numIndices; ++index) {
            this.trees.add(null);
        }
        for (index = 0; index < numIndices; ++index) {
            File f = new File(this.databaseDir);
            final int idx = index;
            String[] files = f.list(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.startsWith("IX" + idx + "V");
                }
            });
            if (files == null) {
                throw new BabuDBException(BabuDBException.ErrorCode.IO_ERROR, "database directory '" + this.databaseDir + "' does not exist");
            }
            int maxView = -1;
            long maxSeq = -1L;
            Pattern p = Pattern.compile(SNAPSHOT_FILENAME_REGEXP);
            for (String fname : files) {
                Matcher m = p.matcher(fname);
                m.matches();
                Logging.logMessage((int)7, (Object)this, (String)("inspecting snapshot: " + fname), (Object[])new Object[0]);
                int view = Integer.valueOf(m.group(2));
                long seq = Long.valueOf(m.group(3));
                if (view > maxView) {
                    maxView = view;
                    maxSeq = seq;
                    continue;
                }
                if (view != maxView || seq <= maxSeq) continue;
                maxSeq = seq;
            }
            try {
                if (maxView > -1) {
                    Logging.logMessage((int)7, (Object)this, (String)("loading database " + this.databaseName + " from latest snapshot:" + this.databaseDir + File.separator + "IX" + index + "V" + maxView + "SEQ" + maxSeq), (Object[])new Object[0]);
                    assert (this.comparators[index] != null);
                    this.trees.set(index, new LSMTree(this.databaseDir + File.separator + LSMDatabase.getSnapshotFilename(index, maxView, maxSeq), this.comparators[index], this.compression, this.maxEntriesPerBlock, this.maxBlockFileSize, !this.disableMMap, this.mmapLimit));
                    this.ondiskLSN = new LSN(maxView, maxSeq);
                    continue;
                }
                this.ondiskLSN = NO_DB_LSN;
                Logging.logMessage((int)7, (Object)this, (String)("no snapshot for database " + this.databaseName), (Object[])new Object[0]);
                assert (this.comparators[index] != null);
                this.trees.set(index, new LSMTree(null, this.comparators[index], this.compression, this.maxEntriesPerBlock, this.maxBlockFileSize, !this.disableMMap, this.mmapLimit));
                continue;
            }
            catch (IOException ex) {
                Logging.logError((int)3, (Object)this, (Throwable)ex);
                throw new BabuDBException(BabuDBException.ErrorCode.IO_ERROR, "cannot load index from disk", ex);
            }
        }
    }

    public LSMTree getIndex(int indexId) {
        assert (indexId >= 0 || indexId < 512);
        return this.trees.get(indexId);
    }

    public int getIndexCount() {
        return this.trees.size();
    }

    public LSN getOndiskLSN() {
        return this.ondiskLSN;
    }

    public int[] createSnapshot() {
        int[] snapIds = new int[this.trees.size()];
        for (int index = 0; index < this.trees.size(); ++index) {
            LSMTree tree = this.trees.get(index);
            snapIds[index] = tree.createSnapshot();
        }
        return snapIds;
    }

    public int[] createSnapshot(int[] indices) {
        int[] snapIds = new int[indices.length];
        for (int index = 0; index < indices.length; ++index) {
            LSMTree tree = this.trees.get(indices[index]);
            snapIds[index] = tree.createSnapshot();
        }
        return snapIds;
    }

    public void writeSnapshot(int viewId, long sequenceNo, int[] snapIds) throws IOException {
        Logging.logMessage((int)6, (Object)this, (String)("writing snapshot, database = " + this.databaseName + "..."), (Object[])new Object[0]);
        for (int index = 0; index < this.trees.size(); ++index) {
            LSMTree tree = this.trees.get(index);
            if (Logging.isInfo()) {
                Logging.logMessage((int)6, (Object)this, (String)("snapshotting index " + index + "(dbName = " + this.databaseName + ")..."), (Object[])new Object[0]);
            }
            File tmpDir = new File(this.databaseDir, ".currentSnapshot");
            File targetDir = new File(this.databaseDir, LSMDatabase.getSnapshotFilename(index, viewId, sequenceNo));
            if (targetDir.exists()) {
                Logging.logMessage((int)7, (Object)this, (String)("skipping index'" + index + ", as a valid checkpoint (" + targetDir + ") exists already"), (Object[])new Object[0]);
                continue;
            }
            if (tmpDir.exists()) {
                FSUtils.delTree((File)tmpDir);
            }
            tree.materializeSnapshot(tmpDir.getAbsolutePath(), snapIds[index]);
            if (!tmpDir.renameTo(targetDir)) {
                throw new IOException("could not rename '" + tmpDir + "' to " + targetDir);
            }
            if (!Logging.isInfo()) continue;
            Logging.logMessage((int)6, (Object)this, (String)("... done (index = " + index + ", dbName = " + this.databaseName + ")"), (Object[])new Object[0]);
        }
        if (Logging.isInfo()) {
            Logging.logMessage((int)6, (Object)this, (String)("snapshot written, database = " + this.databaseName), (Object[])new Object[0]);
        }
    }

    public void writeSnapshot(String directory, int[] snapIds, int viewId, long sequenceNumber) throws IOException {
        for (int index = 0; index < this.trees.size(); ++index) {
            LSMTree tree = this.trees.get(index);
            String newFileName = directory + "/" + LSMDatabase.getSnapshotFilename(index, viewId, sequenceNumber);
            tree.materializeSnapshot(newFileName, snapIds[index]);
        }
    }

    public void writeSnapshot(String directory, int[] snapIds, SnapshotConfig cfg) throws IOException {
        for (int i = 0; i < cfg.getIndices().length; ++i) {
            int index = cfg.getIndices()[i];
            LSMTree tree = this.trees.get(index);
            String newFileName = directory + "/" + LSMDatabase.getSnapshotFilename(index, 0, 0L);
            File dir = new File(directory);
            if (!dir.exists() && !dir.mkdirs()) {
                throw new IOException("Directory doesnt exist and cannot be created:'" + directory + "'");
            }
            tree.materializeSnapshot(newFileName, snapIds[i], index, cfg);
        }
    }

    public void cleanupSnapshot(int viewId, long sequenceNo) throws IOException {
        for (int index = 0; index < this.trees.size(); ++index) {
            LSMTree tree = this.trees.get(index);
            Logging.logMessage((int)6, (Object)this, (String)("linking to snapshot " + this.databaseDir + File.separator + LSMDatabase.getSnapshotFilename(index, viewId, sequenceNo) + ", dbName=" + this.databaseName + ", index=" + index), (Object[])new Object[0]);
            IOException exception = null;
            try {
                tree.linkToSnapshot(this.databaseDir + File.separator + LSMDatabase.getSnapshotFilename(index, viewId, sequenceNo));
            }
            catch (ClosedByInterruptException exc) {
                Logging.logError((int)7, (Object)this, (Throwable)exc);
            }
            catch (IOException exc) {
                Logging.logError((int)3, (Object)this, (Throwable)exc);
                exception = exc;
            }
            Logging.logMessage((int)6, (Object)this, (String)"...done", (Object[])new Object[0]);
            this.ondiskLSN = new LSN(viewId, sequenceNo);
            File f = new File(this.databaseDir);
            String[] files = f.list();
            Pattern p = Pattern.compile(SNAPSHOT_FILENAME_REGEXP);
            for (String fname : files) {
                Matcher m = p.matcher(fname);
                if (!m.matches()) continue;
                int fView = Integer.valueOf(m.group(2));
                int fSeq = Integer.valueOf(m.group(3));
                if (fView >= viewId && (fView != viewId || (long)fSeq >= sequenceNo)) continue;
                File snap = new File(this.databaseDir + File.separator + fname);
                if (snap.isDirectory()) {
                    FSUtils.delTree((File)snap);
                    continue;
                }
                snap.delete();
            }
            if (exception == null) continue;
            throw new IOException(exception);
        }
    }

    public String getDatabaseName() {
        return this.databaseName;
    }

    public static String getSnapshotFilename(int indexId, int viewId, long sequenceNo) {
        return "IX" + indexId + "V" + viewId + "SEQ" + sequenceNo + ".idx";
    }

    public static LSN getSnapshotLSNbyFilename(String fname) {
        Matcher m = Pattern.compile(SNAPSHOT_FILENAME_REGEXP).matcher(new File(fname).getName());
        m.matches();
        return new LSN(Integer.valueOf(m.group(2)), Integer.valueOf(m.group(3)).intValue());
    }

    public static boolean isSnapshotFilename(String fileName) {
        return new File(fileName).getName().matches(SNAPSHOT_FILENAME_REGEXP);
    }

    public ArrayList<DBFileMetaData> getLastestSnapshotFiles() {
        ArrayList<DBFileMetaData> result = new ArrayList<DBFileMetaData>();
        for (int index = 0; index < this.numIndices; ++index) {
            final int idx = index;
            File f = new File(this.databaseDir);
            String[] files = f.list(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.startsWith("IX" + idx + "V");
                }
            });
            int maxView = -1;
            int maxSeq = -1;
            Pattern p = Pattern.compile(SNAPSHOT_FILENAME_REGEXP);
            for (String fname : files) {
                Matcher m = p.matcher(fname);
                m.matches();
                Logging.logMessage((int)7, (Object)this, (String)("inspecting snapshot: " + fname), (Object[])new Object[0]);
                int view = Integer.valueOf(m.group(2));
                int seq = Integer.valueOf(m.group(3));
                if (view > maxView) {
                    maxView = view;
                    maxSeq = seq;
                } else if (view == maxView && seq > maxSeq) {
                    maxSeq = seq;
                }
                if (maxView <= -1) continue;
                String fName = LSMDatabase.getSnapshotFilename(index, maxView, maxSeq);
                File snapshotDir = new File(this.databaseDir + File.separator + fName);
                if (snapshotDir.isDirectory()) {
                    for (File file : snapshotDir.listFiles()) {
                        result.add(new DBFileMetaData(this.databaseDir + File.separator + fName + File.separator + file.getName(), file.length()));
                    }
                    continue;
                }
                result.add(new DBFileMetaData(this.databaseDir + File.separator + fName, snapshotDir.length()));
            }
        }
        return result;
    }

    public int getDatabaseId() {
        return this.databaseId;
    }

    public static final class DBFileMetaData {
        public final String file;
        public final long size;

        public DBFileMetaData(String file, long size) {
            this.file = file;
            this.size = size;
        }
    }
}

