/*
 * Decompiled with CFR 0.152.
 */
package org.xtreemfs.foundation;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.util.Calendar;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.xtreemfs.foundation.LifeCycleThread;
import org.xtreemfs.foundation.TimeServerClient;
import org.xtreemfs.foundation.logging.Logging;

public final class TimeSync
extends LifeCycleThread {
    private static final int MAX_RTT = 1000;
    private TimeServerClient timeServerClient;
    private volatile int timeSyncInterval;
    private volatile int localTimeRenew;
    private volatile ExtSyncSource syncSource;
    private InetSocketAddress gpsdAddr;
    private volatile long localSysTime;
    private volatile long currentDrift;
    private volatile boolean quit;
    private volatile long lastSuccessfulSync;
    private long lastSyncAttempt;
    private volatile int syncRTT;
    private volatile boolean syncSuccess;
    private static TimeSync theInstance;
    private final Pattern gpsdDatePattern;
    private Socket gpsdSocket;

    private TimeSync(ExtSyncSource source, TimeServerClient dir, InetSocketAddress gpsd, int timeSyncInterval, int localTimeRenew) {
        super("TSync Thr");
        this.setDaemon(true);
        this.syncSuccess = false;
        this.gpsdDatePattern = Pattern.compile("GPSD,D=(....)-(..)-(..)T(..):(..):(..)\\.(.+)Z");
        this.init(source, dir, gpsd, timeSyncInterval, localTimeRenew);
    }

    public synchronized void init(ExtSyncSource source, TimeServerClient dir, InetSocketAddress gpsd, int timeSyncInterval, int localTimeRenew) {
        this.localTimeRenew = localTimeRenew;
        this.timeSyncInterval = timeSyncInterval;
        this.timeServerClient = dir;
        this.syncSource = source;
        this.gpsdAddr = gpsd;
        if (this.timeServerClient != null && this.timeSyncInterval != 0 && this.localTimeRenew != 0) {
            this.localTimeRenew = 0;
            Logging.logMessage(7, this, "Disabled the periodic local time renew (set local_clock_renewal to 0) and using always the current system time as base since the time will be corrected by synchronizing with the DIR service.", new Object[0]);
        }
        if (source == ExtSyncSource.GPSD) {
            try {
                if (this.gpsdSocket != null) {
                    this.gpsdSocket.close();
                }
                this.gpsdSocket = new Socket();
                this.gpsdSocket.setSoTimeout(2000);
                this.gpsdSocket.setTcpNoDelay(true);
                this.gpsdSocket.connect(this.gpsdAddr, 2000);
            }
            catch (IOException ex) {
                Logging.logMessage(3, this, "cannot connect to GPSd: " + ex, new Object[0]);
                this.gpsdSocket = null;
            }
        }
    }

    @Override
    public void run() {
        theInstance = this;
        this.notifyStarted();
        String tsStatus = this.localTimeRenew == 0 ? "using the local clock" : "using the local clock (precision is " + this.localTimeRenew + " ms)";
        if (this.timeServerClient != null && this.timeSyncInterval != 0) {
            tsStatus = tsStatus + " and remote sync every " + this.timeSyncInterval + " ms";
        }
        Logging.logMessage(6, Logging.Category.lifecycle, this, "TimeSync is running %s", tsStatus);
        while (!this.quit) {
            long timeBetweenUpdates;
            long previousLocalSysTime = this.localSysTime;
            this.localSysTime = System.currentTimeMillis();
            if (this.localTimeRenew > 0 && previousLocalSysTime != 0L && (timeBetweenUpdates = Math.abs(this.localSysTime - previousLocalSysTime)) > (long)(4 * this.localTimeRenew)) {
                Logging.logMessage(4, this, "The granularity of the renewed local time could not be guaranteed since it took longer to retrieve the latest local time (%d ms) than configured (local_clock_renewal = %d). Maybe the system is under high I/O load and therefore scheduling threads takes longer than usual?", timeBetweenUpdates, this.localTimeRenew);
            }
            if (this.timeSyncInterval != 0 && this.localSysTime - this.lastSyncAttempt > (long)this.timeSyncInterval) {
                this.resync();
            }
            if (this.quit) continue;
            try {
                long sleepTimeMs;
                long l = sleepTimeMs = this.localTimeRenew != 0 ? (long)this.localTimeRenew : (long)this.timeSyncInterval;
                if (sleepTimeMs == 0L) {
                    sleepTimeMs = 600000L;
                }
                TimeSync.sleep(sleepTimeMs);
            }
            catch (InterruptedException ex) {
                break;
            }
        }
        this.notifyStopped();
        this.syncSuccess = false;
        theInstance = null;
    }

    public static TimeSync initialize(TimeServerClient dir, int timeSyncInterval, int localTimeRenew) throws Exception {
        if (theInstance != null) {
            Logging.logMessage(4, Logging.Category.lifecycle, null, "time sync already running", new Object[0]);
            return theInstance;
        }
        TimeSync s = new TimeSync(ExtSyncSource.XTREEMFS_DIR, dir, null, timeSyncInterval, localTimeRenew);
        s.start();
        s.waitForStartup();
        return s;
    }

    public static TimeSync initializeLocal(int localTimeRenew) {
        if (theInstance != null) {
            Logging.logMessage(4, Logging.Category.lifecycle, null, "time sync already running", new Object[0]);
            return theInstance;
        }
        TimeSync s = new TimeSync(ExtSyncSource.LOCAL_CLOCK, null, null, 0, localTimeRenew);
        s.start();
        return s;
    }

    public static TimeSync initializeGPSD(InetSocketAddress gpsd, int timeSyncInterval, int localTimeRenew) {
        if (theInstance != null) {
            Logging.logMessage(4, Logging.Category.lifecycle, null, "time sync already running", new Object[0]);
            return theInstance;
        }
        TimeSync s = new TimeSync(ExtSyncSource.GPSD, null, gpsd, timeSyncInterval, localTimeRenew);
        s.start();
        return s;
    }

    public void close() {
        this.shutdown();
        try {
            this.waitForShutdown();
        }
        catch (Exception e) {
            Logging.logError(3, null, e);
        }
    }

    @Override
    public void shutdown() {
        this.quit = true;
        this.interrupt();
        if (this.gpsdSocket != null) {
            try {
                this.gpsdSocket.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public static long getLocalSystemTime() {
        TimeSync ts = TimeSync.getInstance();
        if (ts.localTimeRenew == 0 || ts.localSysTime == 0L) {
            return System.currentTimeMillis();
        }
        return ts.localSysTime;
    }

    public static long getGlobalTime() {
        TimeSync ts = TimeSync.getInstance();
        if (ts.localTimeRenew == 0 || ts.localSysTime == 0L) {
            return System.currentTimeMillis() + ts.currentDrift;
        }
        return ts.localSysTime + ts.currentDrift;
    }

    public static long getLocalRenewInterval() {
        return TimeSync.getInstance().localTimeRenew;
    }

    public static int getTimeSyncInterval() {
        return TimeSync.getInstance().timeSyncInterval;
    }

    public static int getSyncRTT() {
        return TimeSync.getInstance().syncRTT;
    }

    public static boolean lastSyncWasSuccessful() {
        return TimeSync.getInstance().syncSuccess;
    }

    public static long getLastSuccessfulSyncTimestamp() {
        return TimeSync.getInstance().lastSuccessfulSync;
    }

    public long getDrift() {
        return this.currentDrift;
    }

    private void resync() {
        switch (this.syncSource) {
            case LOCAL_CLOCK: {
                return;
            }
            case XTREEMFS_DIR: {
                try {
                    long tStart;
                    this.lastSyncAttempt = tStart = System.currentTimeMillis();
                    long oldDrift = this.currentDrift;
                    long globalTime = this.timeServerClient.xtreemfs_global_time_get(null);
                    if (globalTime <= 0L) {
                        return;
                    }
                    long tEnd = System.currentTimeMillis();
                    this.syncRTT = (int)(tEnd - tStart);
                    if (this.syncRTT > 1000) {
                        Logging.logMessage(4, Logging.Category.misc, this, "Ignored time synchronization message because DIR took too long to respond (%d ms)", this.syncRTT);
                        this.syncSuccess = false;
                        return;
                    }
                    this.syncSuccess = true;
                    this.currentDrift = (globalTime += (long)(this.syncRTT / 2)) - tEnd;
                    this.lastSuccessfulSync = tEnd;
                    if (Math.abs(oldDrift - this.currentDrift) <= 5000L || oldDrift == 0L) break;
                    Logging.logMessage(3, Logging.Category.misc, this, "STRANGE DRIFT CHANGE from %d to %d", oldDrift, this.currentDrift);
                }
                catch (Exception ex) {
                    this.syncSuccess = false;
                    ex.printStackTrace();
                }
                break;
            }
            case GPSD: {
                try {
                    long tStart;
                    BufferedReader br = new BufferedReader(new InputStreamReader(this.gpsdSocket.getInputStream()));
                    OutputStream os = this.gpsdSocket.getOutputStream();
                    this.lastSyncAttempt = tStart = System.currentTimeMillis();
                    os.write(new byte[]{100, 10});
                    os.flush();
                    long oldDrift = this.currentDrift;
                    String response = br.readLine();
                    long tEnd = System.currentTimeMillis();
                    Matcher m = this.gpsdDatePattern.matcher(response);
                    Calendar c = Calendar.getInstance();
                    if (!m.matches()) {
                        Logging.logMessage(4, this, "cannot parse GPSd response: %s", response);
                        this.syncSuccess = false;
                        return;
                    }
                    c.set(1, Integer.parseInt(m.group(1)));
                    c.set(2, Integer.parseInt(m.group(2)) - 1);
                    c.set(5, Integer.parseInt(m.group(3)));
                    c.set(11, Integer.parseInt(m.group(4)));
                    c.set(12, Integer.parseInt(m.group(5)));
                    c.set(13, Integer.parseInt(m.group(6)));
                    long globalTime = c.getTimeInMillis();
                    Date d = new Date(globalTime);
                    Logging.logMessage(7, this, "global GPSd time: %d (%d:%d:%d)", c.getTimeInMillis(), d.getHours(), d.getMinutes(), d.getSeconds());
                    this.syncRTT = (int)(tEnd - tStart);
                    Logging.logMessage(7, this, "sync RTT: %d ms", this.syncRTT);
                    this.syncSuccess = true;
                    this.currentDrift = (globalTime += (long)(this.syncRTT / 2)) - tEnd;
                    this.lastSuccessfulSync = tEnd;
                    Logging.logMessage(7, Logging.Category.misc, this, "resync success, drift: %d ms", Math.abs(oldDrift - this.currentDrift));
                    if (Math.abs(oldDrift - this.currentDrift) <= 5000L || oldDrift == 0L) break;
                    Logging.logMessage(3, Logging.Category.misc, this, "STRANGE DRIFT CHANGE from %d to %d", oldDrift, this.currentDrift);
                    break;
                }
                catch (Exception ex) {
                    this.syncSuccess = false;
                    ex.printStackTrace();
                }
            }
        }
    }

    public static TimeSync getInstance() {
        if (theInstance == null) {
            throw new RuntimeException("TimeSync not initialized!");
        }
        return theInstance;
    }

    public static boolean isInitialized() {
        return theInstance != null;
    }

    public static enum ExtSyncSource {
        XTREEMFS_DIR,
        GPSD,
        LOCAL_CLOCK;

    }
}

